%%% Decode from GNURadio
%% Clear all variables and console
clearvars;
clc;

%% Init timer
tic;

%% Define some constants
BPSK_CHIPS = [1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 0];  % IEEE 802.15.4 BPSK bit-to-chip mapping (Sec. 13.2.5)
BPSK_CHIPS = [BPSK_CHIPS; 1-BPSK_CHIPS];  % IEEE 802.15.4 BPSK bit-to-chip mapping (Sec. 13.2.5)
BPSK_ZERO_BPSK_CHIP = BPSK_CHIPS(1, :);
BPSK_ONE_BPSK_CHIP = BPSK_CHIPS(2, :);
BPSK_CHIPS_SPREAD = length(BPSK_CHIPS(1, :));

NUM_OF_MSGS = 9;

NUM_OF_NOISE_SAMPLES = 200;
NUM_OF_SAMPLES_SIGNAL_THRESHOLD = 250;
ADDITIONAL_SAMPLES = 500;

EXPECTED_HDR = uint8([0 0 0 0xe5 0x1b 0x41 0x88]);
EXPECTED_PAYLOAD = uint8([0xaa 0x1a 0xff 0xff 0xfe 0xaf 0x81 0x00 0x17 0x2a 0x33 0x33 0x37 0x33 0x33 0x37 0x33 0x33 0x37 0x33 0x33 0x37]);

BITS_IN_BYTE = 8;

%% Generate input file list
pathPrefix = append('.', filesep, 'files', filesep);

fileMask = '*-355060*.bin';
fileListStruct = dir(append(pathPrefix, fileMask));
fileMask = '*-355560*.bin';
fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];
fileMask = '*-356060*.bin';
fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];
fileMask = '*-356560*.bin';
fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];
fileMask = '*-405560*.bin';
fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];
fileMask = '*-406060*.bin';
fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];
fileMask = '*-406560*.bin';
fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];
fileMask = '*-407060*.bin';
fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];

%% 10m PoC file list
% pathPrefix = append('.', filesep, 'files-10m-PoC', filesep);
% fileMask = '*-607560*.bin';  %% 10m PoC 1/2
% % fileListStruct = dir(append(pathPrefix, fileMask));
% fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];
% fileMask = '*-608060*.bin';  %% 10m PoC 2/2
% fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];

%% 433MHz 10m PoC file list
% pathPrefix = append('.', filesep, 'files-433mhz-10m-PoC', filesep);
% fileMask = '*-658060*.bin';
% % fileListStruct = dir(append(pathPrefix, fileMask));
% fileListStruct = [fileListStruct; dir(append(pathPrefix, fileMask))];

%% Build fileNames
fileNames = {fileListStruct.name};

%% Prepare result collection struct and table headings
resultsAll = cell(length(fileListStruct), 1+3*NUM_OF_MSGS+3);
resultsAllHeadings = cell(1, length(resultsAll(1, :)));
resultsAllHeadings(1) = {'filename'};
resultsAllHeadings(end-2:end) = {'bitsjammed' 'bitsreceivedwrong' 'bitsfixedwrong'};
for counter = 1:NUM_OF_MSGS
    resultsAllHeadings(1+counter) = {sprintf('msgexpected_%d', counter)};
    resultsAllHeadings(1+NUM_OF_MSGS+counter) = {sprintf('msgreceived_%d', counter)};
    resultsAllHeadings(1+2*NUM_OF_MSGS+counter) = {sprintf('msgfixed_%d', counter)};
end

%% Define output file
outputFileName = sprintf("%s-results.csv", datetime("now","Format","uuuu-MM-dd-HHmmss"));
outputFile = append(pathPrefix, outputFileName);

%% Perform everything for each input file
numberOfFiles = length(fileNames);
parfor fileIdx = 1:numberOfFiles
    %% Output metrics/results
    resultsAllPerFile = resultsAll(fileIdx, :);

    resultsMsgsExp = zeros(NUM_OF_MSGS, 32+4, "uint8");
    resultsMsgsRx = zeros(NUM_OF_MSGS, 32+4, "uint8");
    resultsMsgsFixed = zeros(NUM_OF_MSGS, 32+4, "uint8");
    
    resultsMetrics = zeros(NUM_OF_MSGS, 3, "uint32");
    resultsBitsJammed = resultsMetrics(:, 1);
    resultsBitsRxWrong = resultsMetrics(:, 2);
    resultsBitsFixedWrong = resultsMetrics(:, 3);

    inputFile = append(pathPrefix, fileNames{fileIdx});

    %% Define seq. no. and CRC constants here (as per matlab performance warning)
    SEQ_NOS = uint8([0; 1; 2; 3; 4; 5; 6; 7; 8]);
    CRC_VALUES = uint8([0xC4, 0xE1;  0x0D, 0x68;  0x47, 0xFA;  0x8E, 0x73;  0xC2, 0xD6;  0x0B, 0x5F;  0x41, 0xCD;  0x88, 0x44;  0xC8, 0x8F;]);
    CRC_LOW = CRC_VALUES(:, 1);
    CRC_HIGH = CRC_VALUES(:, 2);

    %% read input file
    f = fopen(inputFile,'rb');
    values = fread(f, Inf, 'float');
    demodulatedRx = values(1:2:end) + values(2:2:end)*1i;
    fclose(f);
    
    %% Plot input data
    % plot(real(demodulatedRx(1:end)));
    % hold on;
    % plot(imag(demodulatedRx(1:end)));
    % hold off;
    
    %% Find message/packet offsets
    offsets = zeros(NUM_OF_MSGS, 3);
    noiseLevel = max(abs(demodulatedRx(1:NUM_OF_NOISE_SAMPLES)));
    messageCount = 1;
    inMessage = false;
    
    for iterator = 1 : length(demodulatedRx)-NUM_OF_SAMPLES_SIGNAL_THRESHOLD
        if (messageCount <= NUM_OF_MSGS) & (~inMessage & sum( abs(demodulatedRx(iterator:iterator+NUM_OF_SAMPLES_SIGNAL_THRESHOLD - 1)) > noiseLevel ) >= NUM_OF_SAMPLES_SIGNAL_THRESHOLD )
            offsets(messageCount, 1) = SEQ_NOS(messageCount);
            offsets(messageCount, 2) = max(1, iterator - ADDITIONAL_SAMPLES);
            inMessage = true;
        end
        if (inMessage & sum( abs(demodulatedRx(iterator:iterator+NUM_OF_SAMPLES_SIGNAL_THRESHOLD - 1)) <= noiseLevel ) >= NUM_OF_SAMPLES_SIGNAL_THRESHOLD )
            offsets(messageCount, 3) = min(length(demodulatedRx), iterator + ADDITIONAL_SAMPLES);
            messageCount = messageCount + 1;
            inMessage = false;
        end
    end
    
    offsetsStart = offsets(:, 2);
    offsetsEnd = offsets(:, 3);
    
    %% Loop over all found messages
    for currentMsg = 1:length(offsets)
        %% crop and "normalize"    
        croppedDemodulated = demodulatedRx(offsetsStart(currentMsg) : offsetsEnd(currentMsg));
        [~, amplitude] = cart2pol(real(croppedDemodulated), imag(croppedDemodulated));
        demodulated = sign(real(croppedDemodulated)) .* amplitude;
        demodulated = demodulated > 0;
        
        %% determine discriminators
        sigThreshold = 3*noiseLevel;
        jamEndOffset = offsetsEnd(currentMsg) - 1;
        separationSamples = 50;
        sigOffset = find(abs(croppedDemodulated) > sigThreshold, 1);
        jamThreshold = 1.3*max(abs(croppedDemodulated(sigOffset:sigOffset + NUM_OF_SAMPLES_SIGNAL_THRESHOLD - 1)));
        
        jamOffset = sigOffset - 1 + find(abs(croppedDemodulated(sigOffset:end)) > jamThreshold, 1);
        
        % sigMean = mean(abs(croppedDemodulated(sigOffset:jamOffset-separationSamples)));
        % sigMedian = median(abs(croppedDemodulated(sigOffset:jamOffset-separationSamples)));
        sigMax = max(abs(croppedDemodulated(sigOffset:jamOffset-separationSamples)));
        % sigMin = min(abs(croppedDemodulated(sigOffset:jamOffset-separationSamples)));
        
        unjammedRx = abs(croppedDemodulated(jamOffset:end)) < jamThreshold;
        consecUnderThreshold = 100;
        for idx = 1:length(unjammedRx)-consecUnderThreshold
            if sum(unjammedRx(idx:idx+consecUnderThreshold-1)) == consecUnderThreshold
                jamEndOffset = idx + jamOffset - 2;  % two 1-indexed offsets
                break;
            end
        end
    
        % jamMean = mean(abs(croppedDemodulated(jamOffset+0:jamEndOffset+1)));
        jamMax = max(abs(croppedDemodulated(jamOffset+0:jamEndOffset+1)));
        % jamMin = min(abs(croppedDemodulated(jamOffset+0:jamEndOffset+1)));
                
        fixedRx = croppedDemodulated;
        
        ampData = .97*sigMax;
        ampJam = jamMax - 0.5*ampData;
        phiData = [0 pi];
        
        for idex = jamOffset:jamEndOffset
            [~, ampRx] = cart2pol(real(croppedDemodulated(idex)), imag(croppedDemodulated(idex)));
            phiJam = phiData - acos((ampRx^2 - ampData^2 - ampJam^2)/(2*ampData*ampJam));
            if abs(phiJam(1) - cart2pol(real(croppedDemodulated(idex)), imag(croppedDemodulated(idex)))) > abs(phiJam(2) - cart2pol(real(croppedDemodulated(idex)), imag(croppedDemodulated(idex))))
                fixedRx(idex) = +1;
            else
                fixedRx(idex) = -1;
            end
        end
        
        %% "normalize" fixed
        fixedDemodulated = real(fixedRx) > 0;
        
        %% find potential start of phr
        chipBitsThreshold = 11;
        corrOffsets = zeros(length(demodulated), 1);
        fixedCorrOffsets = zeros(length(fixedDemodulated), 1);
        for startOffsetIdx = 1:length(demodulated)-BPSK_CHIPS_SPREAD
            zeroSum = sum(demodulated(startOffsetIdx:startOffsetIdx+BPSK_CHIPS_SPREAD-1)'  == BPSK_ZERO_BPSK_CHIP);
            oneSum = sum(demodulated(startOffsetIdx:startOffsetIdx+BPSK_CHIPS_SPREAD-1)'  == BPSK_ONE_BPSK_CHIP);
            fixedZeroSum = sum(fixedDemodulated(startOffsetIdx:startOffsetIdx+BPSK_CHIPS_SPREAD-1)'  == BPSK_ZERO_BPSK_CHIP);
            fixedOneSum = sum(fixedDemodulated(startOffsetIdx:startOffsetIdx+BPSK_CHIPS_SPREAD-1)'  == BPSK_ONE_BPSK_CHIP);
            if oneSum >= chipBitsThreshold
                corrOffsets(startOffsetIdx) = -1*oneSum;
            elseif zeroSum >= chipBitsThreshold
                corrOffsets(startOffsetIdx) = zeroSum;
            end
            if fixedOneSum >= chipBitsThreshold
                fixedCorrOffsets(startOffsetIdx) = -1*fixedOneSum;
            elseif fixedZeroSum >= chipBitsThreshold
                fixedCorrOffsets(startOffsetIdx) = fixedZeroSum;
            end
        end
        
        chunked = reshape(corrOffsets(1:end-mod(length(corrOffsets), BPSK_CHIPS_SPREAD)), BPSK_CHIPS_SPREAD, []);
        [~, bar] = max(sum(abs(chunked') == 15));
        movingOffset = false;
    
        % check for 'moving' offset
        chunkedSigOffset = floor(sigOffset / length(chunked(:, 1)));
        chunkedBeforeJamOffset = floor((jamOffset - 1) / length(chunked(:, 1)));
        chunkedAfterJamOffset  = ceil((jamEndOffset) / length(chunked(:, 1)));
        [~, beforeOffset] = max( sum( abs(chunked( :, chunkedSigOffset : chunkedBeforeJamOffset ))' == 15 ) );
        [~,  afterOffset] = max( sum( abs(chunked( :, chunkedAfterJamOffset : end) )' == 15 ) );
        if (beforeOffset ~= afterOffset)
            % offsets moves, first use beforeOffset
            bar = beforeOffset;
            movingOffset = true;
            fprintf("Message ID %d: offset shifts from %d to %d, ", currentMsg, beforeOffset, afterOffset);
        end
    
        % find approximate position of shift, assuming in jammed part
        windowSize = 10;
        if (movingOffset)
            for slidingWindow = chunkedBeforeJamOffset : chunkedAfterJamOffset - windowSize
                beforeSum = sum( abs( chunked( beforeOffset, slidingWindow : slidingWindow+windowSize ) ) );
                afterSum  = sum( abs( chunked( afterOffset,  slidingWindow : slidingWindow+windowSize ) ) );
                if (beforeSum < afterSum)
                    % select middle to insert/remove data points for alignment
                    offsetDiff = beforeOffset - afterOffset;
                    dataPoint = (slidingWindow + ceil(windowSize / 2) + 1) * BPSK_CHIPS_SPREAD;
                    if (offsetDiff > 0)
                        demodulated(dataPoint+offsetDiff:end) = demodulated(dataPoint:end-offsetDiff);
                        fixedDemodulated(dataPoint+offsetDiff:end) = fixedDemodulated(dataPoint:end-offsetDiff);
                        fprintf("copying from data point %d onwards and shifting all by %d (discarding the last)\n", dataPoint, offsetDiff);
                    else
                        demodulated(dataPoint+offsetDiff:end+offsetDiff) = demodulated(dataPoint:end);
                        fixedDemodulated(dataPoint+offsetDiff:end+offsetDiff) = fixedDemodulated(dataPoint:end);
                        fprintf("overwriting by shifting from data point %d onwards by %d (duplicating the last)\n", dataPoint, offsetDiff);
                    end
                    break;
                end
            end
        end

        maximumPreambleBits = 24;
        preambleBitsMatch = 24;
    
        fooBar = find((chunked(bar,:)) >= chipBitsThreshold, 60); % "positive" chips
        barFoo = find(chunked(bar, :) <= -chipBitsThreshold, 60); % "negative" chips

        fooFoo = fooBar(1+preambleBitsMatch:length(fooBar)) - fooBar(1:length(fooBar)-preambleBitsMatch);
        barBar = barFoo(1+preambleBitsMatch:length(barFoo)) - barFoo(1:length(barFoo)-preambleBitsMatch);
        fooFooIndex = find(fooFoo == preambleBitsMatch, 1, "last");
        barBarIndex = find(barBar == preambleBitsMatch, 1, "last");
    
        if isempty(fooFooIndex)
            theOffsetIndex = barFoo(barBarIndex-(maximumPreambleBits-preambleBitsMatch));
        else 
            theOffsetIndex = fooBar(fooFooIndex-(maximumPreambleBits-preambleBitsMatch));
        end
        
        phrOffset = theOffsetIndex*BPSK_CHIPS_SPREAD + bar;
        
        %% select offset
        rcvBits = zeros(floor((length(demodulated)-phrOffset+1)/BPSK_CHIPS_SPREAD), 1);
        fixedRcvBits = zeros(floor((length(fixedDemodulated)-phrOffset+1)/BPSK_CHIPS_SPREAD), 1);
        
        for rcvIdx = 1:floor((length(demodulated)-phrOffset+1)/BPSK_CHIPS_SPREAD)
          thisChip = demodulated(phrOffset+(rcvIdx-1)*BPSK_CHIPS_SPREAD : phrOffset+(rcvIdx)*BPSK_CHIPS_SPREAD-1);
          fixedThisChip = fixedDemodulated(phrOffset+(rcvIdx-1)*BPSK_CHIPS_SPREAD : phrOffset+(rcvIdx)*BPSK_CHIPS_SPREAD-1);
          % find the bit corresponding to the closest chip sequence:
          [~, rcvBits(rcvIdx)] = min(sum(xor(thisChip, BPSK_CHIPS')));
          [~, fixedRcvBits(rcvIdx)] = min(sum(xor(fixedThisChip, BPSK_CHIPS')));
        end
        rcvBits = rcvBits - 1; % map 1-based indexing to 0s and 1s
        fixedRcvBits = fixedRcvBits - 1; % map 1-based indexing to 0s and 1s
        
        %% Differential decoding
        diffDec = comm.DifferentialDecoder();
        diffRcvBits = diffDec(rcvBits);
        fixedDiffDec = comm.DifferentialDecoder();
        fixedDiffRcvBits = fixedDiffDec(fixedRcvBits);
        
        %% pack single bits into bytes, build expected message
        byteStartOffset = 0;

        if (sum(diffRcvBits(1+8*3:1+8*4) > 0) == 0)
            byteStartOffset = byteStartOffset + 8;
        end
        
        diffRcvBytes = uint8(zeros(1, length(resultsMsgsRx(currentMsg, :))));
        fixedDiffRcvBytes = uint8(zeros(1, length(resultsMsgsFixed(currentMsg, :))));

        cutOff = mod(length(diffRcvBits(1+byteStartOffset:end)), BITS_IN_BYTE);
        diffRcvBytesPre = uint8(bit2int(diffRcvBits(1+byteStartOffset:end-cutOff), BITS_IN_BYTE, false))';
        fixedDiffRcvBytesPre = uint8(bit2int(fixedDiffRcvBits(1+byteStartOffset:end-cutOff), BITS_IN_BYTE, false))';

        if (length(diffRcvBytesPre) < length(resultsMsgsRx))
            diffRcvBytes(1:length(diffRcvBytesPre)) = diffRcvBytesPre;
        else
            diffRcvBytes = diffRcvBytesPre((1:length(diffRcvBytes)));
        end

        if (length(fixedDiffRcvBytesPre) < length(resultsMsgsFixed))
            fixedDiffRcvBytes(1:length(fixedDiffRcvBytesPre)) = fixedDiffRcvBytesPre;
        else
            fixedDiffRcvBytes = fixedDiffRcvBytesPre((1:length(fixedDiffRcvBytes)));
        end
        
        resultsMsgsExp(currentMsg, :) = [EXPECTED_HDR, offsets(currentMsg, 1), EXPECTED_PAYLOAD, CRC_LOW(currentMsg), CRC_HIGH(currentMsg), zeros(1,4)];
        resultsMsgsRx(currentMsg, :) = diffRcvBytes(1:length(resultsMsgsRx(currentMsg, :)));
        resultsMsgsFixed(currentMsg, :) = fixedDiffRcvBytes(1:length(resultsMsgsFixed(currentMsg, :)));
    
        %% calculate metrics
        jammedBits = chunkedAfterJamOffset - chunkedBeforeJamOffset-1;
        wrongOriBits = sum(sum(dec2bin(bitxor(diffRcvBytes(1:length(resultsMsgsExp(currentMsg, :))), resultsMsgsExp(currentMsg, :))) == '1'));
        wrongFixedBits = sum(sum(dec2bin(bitxor(fixedDiffRcvBytes(1:length(resultsMsgsExp(currentMsg, :))), resultsMsgsExp(currentMsg, :))) == '1'));

        % produce invalid values if something failed
        if (isempty(jammedBits))
            jammedBits = 255;
        end
        if (isempty(wrongOriBits))
            wrongOriBits = 255;
        end
        if (isempty(wrongFixedBits))
            wrongFixedBits = 255;
        end

        resultsBitsJammed(currentMsg) = jammedBits;
        resultsBitsRxWrong(currentMsg) = wrongOriBits;
        resultsBitsFixedWrong(currentMsg) = wrongFixedBits;
    
        %% print results
        fprintf("File %s:\n  Results for message %d (seq. no. 0x%02X):\n", fileNames{fileIdx}, currentMsg, offsets(currentMsg, 1));
    
        fprintf("    Expexted:  ");
        fprintf("%02X", resultsMsgsExp(currentMsg, :));
        fprintf("\n");
    
        fprintf("    Rx:        ");
        fprintf("%02X", resultsMsgsRx(currentMsg, :));
        fprintf("\n");
    
        fprintf("    Fixed Rx:  ");
        fprintf("%02X", resultsMsgsFixed(currentMsg, :));
        fprintf("\n");
        
        fprintf("    jammed bits in oriRx: %d", jammedBits);
        fprintf("\n");
        fprintf("    wrong bits in oriRx:  %d", wrongOriBits);
        fprintf("\n");
        fprintf("    wrong bits in fixRx:  %d", wrongFixedBits);
        fprintf("\n\n");
    
    end
    
    %% Collect results
    resultsMetrics(:, 1) = resultsBitsJammed;
    resultsMetrics(:, 2) = resultsBitsRxWrong;
    resultsMetrics(:, 3) = resultsBitsFixedWrong;
    
    %% Transform results
    resultsAllPerFile(1) = {fileNames(fileIdx)};

    for i = 1:length(resultsMsgsExp(:, 1))
        resultsAllPerFile(1+i) = {sprintf("%02X", resultsMsgsExp(i,:))};
        resultsAllPerFile(1+NUM_OF_MSGS+i) = {sprintf("%02X", resultsMsgsRx(i,:))};
        resultsAllPerFile(1+2*NUM_OF_MSGS+i) = {sprintf("%02X", resultsMsgsFixed(i,:))};
    end
            
    resultsAllPerFile(end-2) = {resultsMetrics(:, 1)};
    resultsAllPerFile(end-1) = {resultsMetrics(:, 2)};
    resultsAllPerFile(end) = {resultsMetrics(:, 3)};
    
    %% Propagate results
    resultsAll(fileIdx, :) = resultsAllPerFile;

end

%% Write results to CSV output file
resultsAllTable = cell2table(resultsAll, 'VariableNames', resultsAllHeadings);
writetable(resultsAllTable, outputFile);

%% Print elapsed time
toc;

