
%% Clear command window and variables
clearvars;
clc;

%% Start timer
tic;

%% Define parameters for evaluation
JAM_MODES = 2; % 1: message-bit jamming, 2: message- and chip-bit jamming
SJR_LO_VALUE = 15; % lowest (absolute) SJR value
SJR_SP_VALUE = 5;  % SJR step
SJR_HI_VALUE = 30; % highest (absolute) SJR value
CHIP_SHIFT_LO = 0; % lowest chip misalignment
CHIP_SHIFT_HI = 3; % highest chip misalignment
TOTAL_RUNS = 1000; % number of runs per simulation setting
MSG_METRICS = 4; % number of "message" types for result csv
TX_OFFSET = 1;   % csv offset for number of transmitted bits
JX_OFFSET = 2;   % csv offset for number of jammed bits
RX_OFFSET = 3;   % csv offset for number of received bits being wrong
FX_OFFSET = 4;   % csv offset for number of reconstructed bits being wrong
PATH_PREFIX = append('.', filesep, 'sim', filesep);  % prefix for csv files output folder, must exist

%% Set number of symbols to transmit
NO_OF_SYMBOLS = (2^3)*64;
NO_OF_EXCESS_JAM_BITS = 56;
NO_OF_TOTAL_JAM_BITS = NO_OF_SYMBOLS+NO_OF_EXCESS_JAM_BITS;
NO_OF_UNJAMMED_SAMPLES = 1000; % Amount of sampled waveform values to be considered unjammed (e.g. SHR)

%% Set power levels
DATA_AMPLITUDE = 10; % Data signal amplitude as absolute value
DATA_LEVEL = pow2db(DATA_AMPLITUDE); % Data signal power level (in dB)
TARGET_NOISE_LEVEL = -30;  % Simulated channel noise (in dB), 30 dB seems a reasonable value: https://ieeexplore.ieee.org/document/9860403

%% Set modulation parameters
NO_OF_CONSTELLATION_POINTS = 2; % number of constellation points in (B)PSK
POSSIBLE_BITS = 0 : (NO_OF_CONSTELLATION_POINTS - 1); % all possible input bits
BPSK_CHIPS = [1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 0];  % IEEE 802.15.4 BPSK bit-to-chip mapping (Sec. 13.2.5)
BPSK_CHIPS = [BPSK_CHIPS; 1-BPSK_CHIPS];  % IEEE 802.15.4 BPSK bit-to-chip mapping (Sec. 13.2.5)
BPSK_CHIPS_SPREAD = length(BPSK_CHIPS(1, :));

%% Define results set
allResults = zeros(JAM_MODES, int8(SJR_HI_VALUE/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), CHIP_SHIFT_HI-CHIP_SHIFT_LO + 1, TOTAL_RUNS, MSG_METRICS);
allResultsMetrics = cell(1, 9);
resultLineCounter = 1;

%% Choose how jamming bits/chips are generated
%%% 0 (message-bit jamming): generate random bits => diff-encode into bits => map bits to chips => filter and send signal
%%% 1 (chip-bit jamming):    generate random chips => filter and send signal
for jamMode = 1:JAM_MODES
    %% Choose SJR values
    for sjr = SJR_LO_VALUE:SJR_SP_VALUE:SJR_HI_VALUE
        %% Choose chip misalignment/shift values
        for chipShift = CHIP_SHIFT_LO:CHIP_SHIFT_HI
            runResultsTx = allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, TX_OFFSET);
            runResultsJx = zeros(1, TOTAL_RUNS);
            runResultsRx = allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, RX_OFFSET);
            runResultsFx = allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, FX_OFFSET);
            fprintf("Run %d.%d.%d.\n", jamMode, sjr, chipShift);
            % use parallel processing
            parfor numberOfRun = 1:TOTAL_RUNS
            % or use single-threaded processing
            % for numberOfRun = 1:TOTAL_RUNS
                %% Set SJR
                TARGET_RELATIVE_JAMMING_POWER = sjr; % The jammer's power value (in dB) compared to signal power (in dB); jammer must be at least 15 dB stronger than signal
                JAMMER_AMPLITUDE = db2pow(DATA_LEVEL + 0.5*TARGET_RELATIVE_JAMMING_POWER); %% must be greater than DATA_AMPLITUDE
                JAMMING_GENERATION_MODE = jamMode - 1;
                CONSTELLATION_POINTS = (2*POSSIBLE_BITS*pi)/(NO_OF_CONSTELLATION_POINTS); % constellation points of modulation scheme
                
                %% Set attacker signal misalignments
                ATTACKER_PHASE_SHIFT = 0*pi; % choose a value from [-pi; pi[, resulting in 0 <= x < 2pi
                ATTACKER_CHIP_SHIFT = chipShift; % choose misalignment of sampled jamming chips to data chips, negative means jam earlier
                
                %% Generate random bits and signal points
                dataBits = floor(NO_OF_CONSTELLATION_POINTS.*rand(1,NO_OF_SYMBOLS)); % sample random symbols uniformly from 1:M
                
                %% Generate modulated chips and signals
                RESERVED_BIT = 0;
                SYMBOLS_PER_CHIP = 4; % Default value is 10, as per the example, 4 suffices: https://www.mathworks.com/help/comm/ug/end-to-end-ieee-802-15-4-phy-simulation.html
                
                %%%% 802.15.4 packet structure:
                %%%% ________________________________________________________
                %%%% |    32    |    8     |    7     |    1     |  <=1016  |
                %%%% |---------------------|---------------------|----------|
                %%%% | Preamble |   SFD    |  Length  | Reserved |  Payload |
                %%%% |---------------------|---------------------|----------|
                %%%% |         SHR         |         PHR         |   MPDU   |
                %%%% |---------------------|---------------------|----------|
                %%%% |         40          |          8          |  <=1016  |
                %%%% |_____________________|_____________________|__________|
                
                %%%%%%% Ripped & adapted from:
                %%%%%%% matlabshared.supportpkg.getSupportPackageRoot()+'\toolbox\comm\supportpackages\zigbee\+lrwpan\PHYGeneratorBPSK.m'
                %%%%%%% Includes some customizations
                % Oversampling rate (OSR) specification
                OSR = SYMBOLS_PER_CHIP;
                
                %% Synchronization header (SHR)
                % Preamble is 4 octets, all set to 0.
                preamble = zeros(4*8, 1);
                % Start-of-frame delimiter (SFD)
                SFD = [1 1 1 0 0 1 0 1]'; % value from standard (see Fig. 68, IEEE 802.15.4, 2011 Revision)
                SHR = [preamble; SFD];
                
                %% PHY Header (PHR)
                frameLength = int2bit(length(dataBits')/8, 7, false);
                PHR = [frameLength; RESERVED_BIT];
                
                %% PHY protocol data unit:
                PPDU = [SHR; PHR; dataBits'];
                
                %% Differential Encoding
                diffEnc = comm.DifferentialEncoder();
                diffDataBits = diffEnc(PPDU);
                
                %% Bit to chip mapping
                % chips = [1 1 1 1 0 1 0 1 1 0 0 1 0 0 0;  % Chip for 0 bit
                %          0 0 0 0 1 0 1 0 0 1 1 0 1 1 1]; % Chip for 1 bit
                dataChipBits = zeros(length(BPSK_CHIPS), length(diffDataBits)); % preallocate
                for idx = 1:length(diffDataBits)
                  dataChipBits(:, idx) = BPSK_CHIPS(1+diffDataBits(idx), :); % +1 for 1-based indexing
                end
                
                %% BPSK modulation
                iniPhase = pi; % 1 for positive pulse, 0 for negative pulse
                modulatedDataBits = pskmod(dataChipBits(:), NO_OF_CONSTELLATION_POINTS, iniPhase, 'gray');  % generate phase values, use iniPhase as phase offset and 'gray' encoding
                
                %% Filtering
                rcosfilt = comm.RaisedCosineTransmitFilter('RolloffFactor', 1, 'OutputSamplesPerSymbol', OSR, 'Shape', 'Normal');
                filterDelay = max(rcosfilt.grpdelay);
                % append some zeros due to filter delay, flushing out all bits
                modulatedDataBits = [modulatedDataBits; zeros(floor(filterDelay/OSR), 1)];
                dataWaveform = DATA_AMPLITUDE * rcosfilt( modulatedDataBits );
                
                %% Add white Guassian noise to data signal
                noisyDataWaveform = awgn(dataWaveform, -TARGET_NOISE_LEVEL, 'measured');
                %%%%%%% :morf deppiR
                
                switch JAMMING_GENERATION_MODE
                    case 0 % message-bit jamming/properly generated random jamming bits
                        %% Simulating randomly chosen jamming bits
                        jamBits = floor(NO_OF_CONSTELLATION_POINTS.*rand(1,NO_OF_TOTAL_JAM_BITS)); % sample random symbols uniformly from 1:M
                        %% Diff-encode jamming bits
                        diffEnc.release();
                        diffEnc.reset();
                        diffJamBits = diffEnc(jamBits');
                        %% Map encoded jamming bits to BPSK chips
                        jamChipBits = zeros(length(BPSK_CHIPS), length(diffJamBits)); % preallocate
                        for idx = 1:length(diffJamBits)
                          jamChipBits(:, idx) = BPSK_CHIPS(1+diffJamBits(idx), :); % +1 for 1-based indexing
                        end
                    case 1 % chip-bit jamming
                        %% Simulating randomly chosen jamming chips
                        jamChipBits = reshape(floor(NO_OF_CONSTELLATION_POINTS.*rand(1,(NO_OF_SYMBOLS+NO_OF_EXCESS_JAM_BITS)*BPSK_CHIPS_SPREAD)), 15, []); % sample random symbols uniformly from 1:M
                end
                
                %%% Modulate jamming bits
                modulatedJamBits = pskmod(jamChipBits(:), NO_OF_CONSTELLATION_POINTS, iniPhase+ATTACKER_PHASE_SHIFT, 'gray');  % generate phase values, use iniPhase as phase offset and 'gray' encoding
                
                %% Filter and generate waveform
                rcosfilt.reset();
                modulatedJamBits = [modulatedJamBits; zeros(floor(filterDelay/OSR), 1)]; % flush out remaining bits due to filter delay
                jamWaveform = JAMMER_AMPLITUDE * rcosfilt(modulatedJamBits);
                modulatedPhrSize = (length(BPSK_CHIPS(1, :)) * SYMBOLS_PER_CHIP) * (length(SHR) + length(PHR));
                
                % pre-allocate jammedDataWaveform
                jammedDataWaveform = zeros(length(noisyDataWaveform), 1);
                % copy (unjammed) PHR
                jammedDataWaveform(1 : modulatedPhrSize+ATTACKER_CHIP_SHIFT) = noisyDataWaveform(1 : modulatedPhrSize+ATTACKER_CHIP_SHIFT);
                % "jam" data signal by copying the sum of both waveforms
                jammedDataWaveform(1 + modulatedPhrSize + ATTACKER_CHIP_SHIFT : length(jammedDataWaveform)) = noisyDataWaveform( 1 + modulatedPhrSize + ATTACKER_CHIP_SHIFT : length(noisyDataWaveform), 1) ...
                    + jamWaveform( 1 : length(noisyDataWaveform) - (modulatedPhrSize + ATTACKER_CHIP_SHIFT) );
                
                
                %% Plot some samples
                % limiter = min([length(jamWaveform)+modulatedPhrSize length(jammedDataWaveform) length(dataWaveform)]);
                % if (limiter > 10000)
                %     limiter = 10000;
                % end
                % jamWaveformPlot = plot(real(jammedDataWaveform(1:limiter)));
                % hold on;
                jammingThreshold = 1.1*max(abs(real(jammedDataWaveform(1:NO_OF_UNJAMMED_SAMPLES))));
                % jammingThrPlot = plot([0 limiter], [jammingThreshold jammingThreshold], 'Color', [0.93 0.69 0.13]);
                % hold off;
                
                %% Reconstruct original data
                % Find beginning of jamming waveform...
                %   startOfJammedOffset = modulatedPhrSize + filterDelay + ATTACKER_CHIP_SHIFT + 1; % +1 for pointing to start index of data
                % ... by finding first 3 consecutive values above defined threshold, abusing max function, ignoring last couple of values
                [ ~, startOfJammedOffset] = max(...
                    abs(real(jammedDataWaveform(1:end-4, :))) > jammingThreshold ...
                    & abs(real(jammedDataWaveform(2:end-3, :))) > jammingThreshold ...
                    & abs(real(jammedDataWaveform(3:end-2, :))) > jammingThreshold ...
                    );
                
                %% find jamming offset
                % Sometimes, the jamming threshold is passed one sample earlier, increase by one seems to be working
                startOfJammedOffset = startOfJammedOffset + 1;
                
                %% calculate amplitudes
                sigMax = max(abs(jammedDataWaveform(1:NO_OF_UNJAMMED_SAMPLES)));
                jamMax = max(abs(jammedDataWaveform(startOfJammedOffset+1 : startOfJammedOffset+NO_OF_UNJAMMED_SAMPLES)));
                ampData = 0.97*sigMax;
                ampJam = 0.96*jamMax-0.5*ampData;
                
                %% prepare some arrays
                % Pre-allocate array for chip-bit reconstruction based on OSR => windowed
                noOfChipBitSamples = length(jammedDataWaveform(startOfJammedOffset:end, :));
                chipInvertSize = floor(noOfChipBitSamples / OSR);
                if (chipInvertSize*OSR < noOfChipBitSamples)
                    chipInvertSize = chipInvertSize + 1;
                end
                chipInvertArr = zeros(chipInvertSize, 1);
                
                invertIndex = 1; % use custom counter
                for chipWindow = startOfJammedOffset:OSR:length(jammedDataWaveform) - mod(length(jammedDataWaveform), OSR) - OSR % +1 already included, exclude last window, needs special treatment
                    if ( ( max(abs(real(jammedDataWaveform(chipWindow : chipWindow + OSR - 1, :)))) >= jammingThreshold ) )
                        %% calc reconstruction criteria
                        phiData = [pi 0];
                        oversampledPulse = jammedDataWaveform(chipWindow : chipWindow + OSR - 1, :);
                        [~, chosenPulseRep] = max(abs(oversampledPulse));
                        [phiRx, ampRx] = cart2pol(real(oversampledPulse(chosenPulseRep)), imag(oversampledPulse(chosenPulseRep)));
                        phiJam = phiData - acos((ampRx^2 - ampData^2 - ampJam^2)/(2*ampData*ampJam));
            
                        if abs(phiJam(1) - phiRx) > abs(phiJam(2) - phiRx)
                            chipInvertArr(invertIndex) = 0;
                        else
                            chipInvertArr(invertIndex) = 1;
                        end
                    end
                    %% increase counter
                    invertIndex = invertIndex + 1;
                end
                
                % Treat last (potentially incomplete) window separately
                if ( ( max(abs(real(jammedDataWaveform(chipWindow : chipWindow + OSR - 1, :)))) >= jammingThreshold ) )
                    %% calc reconstruction criteria
                    phiData = [pi 0];
                    oversampledPulse = jammedDataWaveform(chipWindow : chipWindow + OSR - 1, :);
                    [~, chosenPulseRep] = max(abs(oversampledPulse));
                    [phiRx, ampRx] = cart2pol(real(oversampledPulse(chosenPulseRep)), imag(oversampledPulse(chosenPulseRep)));
                    phiJam = phiData - acos((ampRx^2 - ampData^2 - ampJam^2)/(2*ampData*ampJam));
        
                    if abs(phiJam(1) - phiRx) > abs(phiJam(2) - phiRx)
                        chipInvertArr(invertIndex) = 0;
                    else
                        chipInvertArr(invertIndex) = 1;
                    end
                end
                invertIndex = invertIndex + 1;
                
                %% Compare data and jamData chips for evaluation
                % Skip SHR and PHR
                dataOffset = length(SHR) + length(PHR) + 1;  % 1-index, points to start of data chip (i.e. beginning of 15-bit chip sequence)
                dataOffsetFlattened = BPSK_CHIPS_SPREAD * (dataOffset-1) + 1; % multiply with spreading factor, but subtract 1 from dataOffset before to get length, add one to point to beginning of first-jammed sequence
                dataChipBitsFlattened = reshape(dataChipBits, [], 1);
                jamChipBitsFlattened =  reshape(jamChipBits, [], 1);
                jamOffset = round(ATTACKER_CHIP_SHIFT/OSR);
                
                %% Decode received waveform
                %%%%%%% Ripped from:
                %%%%%%% matlabshared.supportpkg.getSupportPackageRoot()+'\toolbox\comm\supportpackages\zigbee\+lrwpan\PHYDecoderBPSK.m'
                %%%%%%% Includes some customizations
                rcosrcvfilt = comm.RaisedCosineReceiveFilter('RolloffFactor', 1, ...
                            'InputSamplesPerSymbol', OSR, 'DecimationFactor', OSR, 'Shape', 'Normal');
                rcvDelay = rcosrcvfilt.FilterSpanInSymbols;
                paddedRcvWaveform = [jammedDataWaveform; zeros(rcvDelay*OSR, 1)];
                rcvFiltered = rcosrcvfilt(paddedRcvWaveform);
                rcvFiltered = rcvFiltered(1+rcvDelay:end);
                
                %% BPSK demodulation
                demodulated = pskdemod(rcvFiltered, 2, iniPhase);
                fixDemodulated = demodulated; % copy for post-processing
                
                %% Modify some received demoulated chip bits
                for demodChipBitsIdx = 1 : min(length(fixDemodulated)-dataOffsetFlattened-jamOffset+1, length(chipInvertArr))
                    fixDemodulated(round((startOfJammedOffset - filterDelay) / OSR) + demodChipBitsIdx ) = chipInvertArr(demodChipBitsIdx);
                end
                
                %% Chip to bit mapping for received-as-is and corrected-as-decided
                rcvBits = zeros(length(demodulated)/uint64(BPSK_CHIPS_SPREAD), 1);
                rcvBitsFixed = zeros(length(rcvBits), 1);
                for rcvIdx = 1:length(demodulated)/BPSK_CHIPS_SPREAD
                  thisChip = demodulated(1+(rcvIdx-1)*BPSK_CHIPS_SPREAD : rcvIdx*BPSK_CHIPS_SPREAD);
                  thisChipFixed = fixDemodulated(1+(rcvIdx-1)*BPSK_CHIPS_SPREAD : rcvIdx*BPSK_CHIPS_SPREAD);
                  % find the bit corresponding to the closest chip sequence:
                  [~, rcvBits(rcvIdx)] = min(sum(xor(thisChip, BPSK_CHIPS')));
                  [~, rcvBitsFixed(rcvIdx)] = min(sum(xor(thisChipFixed, BPSK_CHIPS')));
                end
                rcvBits = rcvBits - 1; % map 1-based indexing to 0s and 1s
                rcvBitsFixed = rcvBitsFixed - 1; % map 1-based indexing to 0s and 1s
                
                %% Differential decoding
                diffDec = comm.DifferentialDecoder();
                diffRcvBits = diffDec(rcvBits);
                diffDec.reset();
                diffRcvBitsFixed = diffDec(rcvBitsFixed);
                jammedMPDU = diffRcvBits(dataOffset:end);
                fixedMPDU = diffRcvBitsFixed(dataOffset:end);
                %%%%%%% :morf deppiR
                
                %% Collect results of current run
                runResultsTx(numberOfRun) = length(PPDU(1:dataOffset-1)) + length(jammedMPDU);
                runResultsJx(numberOfRun) = length(jammedMPDU);
                runResultsRx(numberOfRun) = runResultsTx(numberOfRun) - (sum(diffRcvBits(1:dataOffset-1) == PPDU(1:dataOffset-1)) + sum(jammedMPDU == dataBits'));
                runResultsFx(numberOfRun) = runResultsTx(numberOfRun) - (sum(diffRcvBitsFixed(1:dataOffset-1) == PPDU(1:dataOffset-1)) + sum(fixedMPDU == dataBits'));
            end
            %% Collect results of all runs for the current settings
            allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, TX_OFFSET) = squeeze(runResultsTx);
            allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, JX_OFFSET) = squeeze(runResultsJx);
            allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, RX_OFFSET) = squeeze(runResultsRx);
            allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, FX_OFFSET) = squeeze(runResultsFx);

            %% Write results to CSV output file
            pathPrefix = PATH_PREFIX;
            outputFileName = sprintf("%d-%d-%d.csv", jamMode, sjr, chipShift);
            outputFile = append(pathPrefix, outputFileName);
            resultsAllHeadings = {'TXlen' 'Jammed' 'RXwrong' 'FXwrong'};
            resultsPerRun = cell(3, TOTAL_RUNS);
            resultsPerRun(TX_OFFSET, :) = num2cell(uint32(squeeze(allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, TX_OFFSET))));
            resultsPerRun(JX_OFFSET, :) = num2cell(uint32(squeeze(allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, JX_OFFSET))));
            resultsPerRun(RX_OFFSET, :) = num2cell(uint32(squeeze(allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, RX_OFFSET))));
            resultsPerRun(FX_OFFSET, :) = num2cell(uint32(squeeze(allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, :, FX_OFFSET))));

            resultsAllTable = cell2table(resultsPerRun', 'VariableNames', resultsAllHeadings);
            writetable(resultsAllTable, outputFile);

            %% Calculate metrics
            % BRR = |0.5 - BER| / 0.5
            theBer = squeeze(runResultsFx)/squeeze(allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, 1, JX_OFFSET));  % use first MPDU length to get jammed length
            theBrr = abs(0.5-theBer) / 0.5;
            theMean = mean(theBrr);
            theMin = min(theBrr);
            theMax = max(theBrr);
            theRxBer = squeeze(runResultsRx)/squeeze(allResults(jamMode, uint8(sjr/SJR_SP_VALUE-SJR_LO_VALUE/SJR_SP_VALUE + 1), chipShift+1, 1, JX_OFFSET));  % use first MPDU length to get jammed length
            theRxBrr = abs(0.5-theRxBer) / 0.5;
            theRxMean = mean(theRxBrr);
            theRxMin = min(theRxBrr);
            theRxMax = max(theRxBrr);
            allResultsMetrics(resultLineCounter, :) = num2cell([jamMode sjr chipShift theMean theMin theMax theRxMean theRxMin theRxMax]);
            resultLineCounter = resultLineCounter + 1;
        end
    end
end

%% Write resulting metrics of all runs and settings to CSV output file
resultsAllHeadings = {'Mode' 'SJR' 'ChipShift' 'meanFix' 'minFix' 'maxFix' 'meanRx' 'minRx' 'MaxRx'};
pathPrefix = PATH_PREFIX;
metricsOutputFile = append(pathPrefix, sprintf("%s-results.csv", datetime("now","Format","uuuu-MM-dd-HHmmss")));

resultsAllTable = cell2table(num2cell(allResultsMetrics), 'VariableNames', resultsAllHeadings);
writetable(resultsAllTable, metricsOutputFile);

%% Stop timer
toc;

%% END
