"""
Embedded Python Blocks:

Each time this file is saved, GRC will instantiate the first class it finds
to get ports and parameters of your block. The arguments to __init__  will
be the parameters. All of them are required to have default values!
"""

import numpy as np
from gnuradio import gr
import pmt


class blk(gr.basic_block):  # other base classes are basic_block, decim_block, interp_block
    """Embedded Python Block example - a simple multiply const"""

    def __init__(self, delay_by_samples = 7*8*15, jam_length_tag = "jam_len", send_length_tag = "packet_len"):  # only default arguments here
        """arguments to this function show up as parameters in GRC"""
        gr.sync_block.__init__(
            self,
            name='JX Delay',   # will show up in GRC
            in_sig=[np.complex64],
            out_sig=[np.complex64]
        )
        # if an attribute with the same name as a parameter is found,
        # a callback is registered (properties work, too).
        self.delay_by_samples = delay_by_samples
        self.jam_length_tag = jam_length_tag
        self.send_length_tag = send_length_tag
        self.set_tag_propagation_policy(gr.TPP_DONT)  # prevent already present tags from being propagated
        self.delaying = False
        self.delayed = 0
        self.delay_array = np.zeros(self.delay_by_samples, dtype=np.complex64) # generate array of zeros to prepend
        self.lastTagOffset = -1

    def work(self, input_items, output_items):
        in_buff_length = len(input_items[0])
        out_buff_length = len(output_items[0])
        tag_window_length = min(in_buff_length, 4800)
        offset = -1
        readOffset = 0
        read = tag_window_length
        write = 0
        jam_length = 0
        if (self.delaying):
            writable = min(out_buff_length, self.delay_by_samples - self.delayed)
            if (writable > 0):
                output_items[0][:writable] = self.delay_array[:writable]
                write = writable
                self.delayed += writable
            if (self.delayed >= self.delay_by_samples):
                self.delaying = False
                self.delayed = 0
            return write
        # look for packet_len tag, at start of every packet
        tags = self.get_tags_in_window(0, 0, tag_window_length)
        # check for tag
        for tag in tags:
            key = pmt.to_python(tag.key)
            if (key == self.jam_length_tag):
                jam_length = pmt.to_python(tag.value)
                offset = tag.offset - self.nitems_read(0)
                break
        
        if ((offset >= 0) and ((offset + self.nitems_read(0)) != self.lastTagOffset)):
            # write out everything appearing before the tag (i.e. from a previous packet)
            output_items[0][:offset] = input_items[0][:offset]
            write += offset
            readOffset = offset
            self.lastTagOffset = offset + self.nitems_read(0)

            # add the (updated) jam length tag
            self.add_item_tag(0, self.nitems_written(0)+write, pmt.intern(self.send_length_tag), pmt.from_long(self.delay_by_samples+jam_length))
            
            # write zeros (i.e. the delay)
            self.delaying = True

            self.consume_each(offset)
            return write

        # write actual input
        remainingOut = out_buff_length - write
        remainingWrite = min(remainingOut, tag_window_length)
        output_items[0][write : write + remainingWrite] = input_items[0][readOffset : readOffset + remainingWrite]
        write += remainingWrite

        # update processed in-bytes
        read = readOffset + remainingWrite

        # ignore the rest and consume the ckecked window
        self.consume_each(read)
        return write

