"""
Embedded Python Blocks:

Each time this file is saved, GRC will instantiate the first class it finds
to get ports and parameters of your block. The arguments to __init__  will
be the parameters. All of them are required to have default values!
"""

import numpy as np
from gnuradio import gr
import pmt


class blk(gr.basic_block):  # other base classes are basic_block, decim_block, interp_block
    """Embedded Python Block example - a simple multiply const"""

    def __init__(self, no_of_bits = 8*15, length_tag = "packet_len", jam_length_tag = "jam_len", jamming_enabled = False):  # only default arguments here
        """arguments to this function show up as parameters in GRC"""
        gr.sync_block.__init__(
            self,
            name='Gen Jamming Bits',   # will show up in GRC
            in_sig=[np.byte],
            out_sig=[np.byte]
        )
        # if an attribute with the same name as a parameter is found,
        # a callback is registered (properties work, too).
        self.no_of_bits = no_of_bits
        self.length_tag = length_tag
        self.jam_length_tag = jam_length_tag
        self.jamming_enabled = jamming_enabled
        self.scaling_factor = 8*15
        self.rng_generator =  np.random.default_rng()
        self.random_bits = self.rng_generator.integers(0, 2, size=self.no_of_bits, dtype=np.byte)
        self.set_tag_propagation_policy(gr.TPP_DONT)  # prevent already present tags from being propagated

    def work(self, input_items, output_items):
        if (self.jamming_enabled):
            in_buff_length = len(input_items[0])
            tag_window_length = min(in_buff_length, 4800)
            offset = -1
            read = tag_window_length
            write = 0
            # look for packet_len tag, at start of every packet
            tags = self.get_tags_in_window(0, 0, tag_window_length)
            
            # check for tag
            for tag in tags:
                key = pmt.to_python(tag.key)
                if (key == self.length_tag):
                    value = pmt.to_python(tag.value)
                    offset = tag.offset - self.nitems_read(0)
                    # length tag is not scaled, so multiply by 15*8 (spreading and unpacking)
                    read = value * self.scaling_factor
                    break
            
            if (offset >= 0):
                output_items[0][:offset] = input_items[0][:offset]
                output_items[0][offset : offset+self.no_of_bits] = self.random_bits
                write = offset+self.no_of_bits

                # add the jam length tag
                self.add_item_tag(0, self.nitems_written(0), pmt.intern(self.jam_length_tag), pmt.from_long(write))

            # ignore the rest and consume the ckecked window
            self.consume_each(read)
            return write
        else:
            # don't produce any output
            self.consume_each(len(input_items[0]))
            return 0
