"""
Embedded Python Blocks:

Each time this file is saved, GRC will instantiate the first class it finds
to get ports and parameters of your block. The arguments to __init__  will
be the parameters. All of them are required to have default values!
"""

import numpy as np
from gnuradio import gr
import pmt


class blk(gr.basic_block):  # other base classes are basic_block, decim_block, interp_block
    """Embedded Python Block example - a simple multiply const"""

    def __init__(self, phr_len_bytes=6, packet_len_tag="packet_len", mtu=127, sfd=0xe5, lossyAccept=1 ):  # only default arguments here
        """arguments to this function show up as parameters in GRC"""
        gr.sync_block.__init__(
            self,
            name='Detect and write PDU',   # will show up in GRC
            in_sig=[np.byte],
            out_sig=[np.byte]
        )
        # if an attribute with the same name as a parameter is found,
        # a callback is registered (properties work, too).
        self.mtu = mtu
        self.phr_len_bytes = phr_len_bytes
        self.packet_len_tag = packet_len_tag
        self.sfd = sfd # SFD value
        self.preamble_len = 4 # first 4 bytes of SHR
        self.sfd_offset = self.preamble_len # SFD byte at offset 4
        self.phr_offset = self.sfd_offset + 1 # PHR/length "byte" at offset 5
        self.seq_no_offset = self.phr_offset + 3 # Sequence number field should be at offset 8
        self.lossyAccept = lossyAccept # number of first SHR bytes acceptable to be missing, must not be greater than 3
        self.current_packet_length = 0
        self.lost_offset = 0
        self.preamble_content = np.zeros(self.preamble_len)

    def check_preamble_access_code(self, input_items):
        correct_preamble = False
        correct_sfd = False
        self.lost_offset = 0
        if (np.array_equal(input_items[0][:self.preamble_len], self.preamble_content)):
            correct_preamble = True
        elif(np.array_equal(input_items[0][:self.preamble_len-self.lossyAccept], self.preamble_content[:self.preamble_len-self.lossyAccept])):
            self.lost_offset = 1
            correct_preamble = True
        else:
            mismatching = np.nonzero(input_items[0][:self.preamble_len])[0] # returns array per dimension with indices where entries are non-zero
            if ( (len(mismatching) > 0) and (len(mismatching) <= self.lossyAccept) ):
                if ( mismatching[0] > 0 ):
                    correct_preamble = True
        if (np.uint8(input_items[0][self.sfd_offset-self.lost_offset]) == np.uint8(self.sfd)):
            correct_sfd = True
        self.current_packet_length = np.uint8(input_items[0][self.phr_offset-self.lost_offset])
        return (correct_preamble and correct_sfd)

    def work(self, input_items, output_items):
        if (len(input_items[0]) < self.phr_len_bytes):
            return 0
        else:
            if (self.check_preamble_access_code(input_items)):
                if (self.current_packet_length > self.mtu):
                    self.current_packet_length = 20
                if (len(input_items[0]) >= (self.phr_len_bytes+self.current_packet_length-self.lost_offset)):
                    output_items[0][:self.current_packet_length] = input_items[0][self.phr_len_bytes-self.lost_offset:self.current_packet_length+self.phr_len_bytes-self.lost_offset]
                    self.add_item_tag(0, self.nitems_written(0), pmt.intern(self.packet_len_tag), pmt.from_long(self.current_packet_length))
                    self.consume_each(self.phr_len_bytes+self.current_packet_length-self.lost_offset)
                    return self.current_packet_length
                else:
                    return 0
            else:
                self.consume_each(1)
                return 0
        
