function [X, MU, info] = localSQP(gradL, hessL, h, Jh, x0, mu0, tol, maxit)
% [X, MU. info] = localSQP(gradL, hessL, h, Jh, x0, mu0, tol, maxit)
%
% Implementation of local SQP method for equality constrained problems
% of type      min f(x)  s.t. h(x) = 0
%
% INPUT: 
%       gradL --> function handle to the gradient of the Lagrangian function
%       hessL --> function handle to the hessian of the Lagrangian function
%           h --> function handle to the constraint function h
%          Jh --> function handle to the constraint function's Jacobian
%          x0 --> initial guess for the variables
%         mu0 --> initial guess for the lagrangian multipliers
%         tol --> tolerance for KKT conditions
%       maxit --> maximum number of iterations
%
% OUTPUT:   X --> matrix containing k-th iterates in column k
%          MU --> matrix containing k-th multipliers in column k
%      info --> flag indicating whether a KKT point was found
%               info = 0  --> KKT point found
%               info = -1 --> maxit exceeded

% ensure x0 and mu0 are column vectors
x0  = reshape(x0,  [], 1);
mu0 = reshape(mu0, [], 1);

% initialize iterates and info flag
xk = x0;
uk = mu0;
info = -1;  % error value

% query dimensions
dimx = size(xk,1);
dimu = size(uk,1);

% prepare zero matrix for system matrix 
zerou = zeros(dimu, dimu);

% prepare storage for iterates and multipliers
chunksize = max(10, round(maxit/20));
capacity  = chunksize;
X  = zeros(dimx, chunksize);
MU = zeros(dimu, chunksize);

for k = 1:maxit
  
  % increase storage if necessary
  if (k > capacity)
     X(end, end+chunksize) = 0;
     MU(end, end+chunksize) = 0;
     capacity = capacity + chunksize;
  end
  
  % store current iterate
  X(:,k)  = xk;
  MU(:,k) = uk;
   
  % evaluate gradient of Lagrangian and the constraint function
  h_xk     = h(xk);
  gradL_xk = gradL(xk,uk);
  vecRHS   = [gradL_xk ; h_xk];  % vector of rhs
  
  % check if KKT point is found
  if norm(vecRHS) < tol
     info = 0; 
     break; 
  end
  
  % build full matrix (inefficient, but following the notation)
  Hk  = hessL(xk,uk);            % Hessian of Lagrangian
  Jhk = Jh(xk);                  % Jacobian of constraint
  Mk = [ Hk Jhk.' ; Jhk zerou ]; % system matrix
  
  % calculate step (note: the minus is shifted into the updates)
  dk = Mk \ vecRHS;
  
  % update variables (note the minus, see computation of dk above)
  xk = xk - dk(1:dimx);
  uk = uk - dk(dimx+1:end);
 
end

% remove superfluous storage cells
X  = X(:,1:k);
MU = MU(:,1:k);

% finito
return

end